package gov.va.vamf.scheduling.direct.datalayer.appointment;

import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifiers;
import gov.va.vamf.scheduling.direct.datalayer.user.VarPatientCorrelationGateway;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.List;

/**
 * The VAR application supports patient identifiers from multiple assigning authorities (e.g. EDIPI and ICN)
 * This class provides a set of functions to lookup identifiers from the different assigning authorities based on
 * provided patient identifier(s) from a given assigning authority
 */
@Component
public class PatientIdentifiersDataService {
    @Resource
    protected VarPatientCorrelationGateway patientCorrelationGateway;

    /**
     * Retrieves PatientIdentifier from the ICN assigning authority for a patient.
     * Uses the provided PatientIdentifier to lookup and return the ICN identifier.
     * @param patientIdentifier PatientIdentifier from any of the supported assigning authorities
     * @return a PatientIdentifier for the provided patient from the ICN assigning authority
     * @return null if no identifier is found for the ICN assigning authority
     */
    public PatientIdentifier getIcnIdentifier(PatientIdentifier patientIdentifier) {
        PatientIdentifiers correspondIdentifiers = patientCorrelationGateway.getCorrespondIdentifiers(patientIdentifier);
        List<String> icnIdentifiers = getIcnIdentifiers(correspondIdentifiers);

        return ((icnIdentifiers == null || icnIdentifiers.isEmpty()) ? null : new PatientIdentifier("ICN", stripHash(icnIdentifiers.get(0))));
    }

    /**
     * Loops through a provided list of patient identifiers and looks for ICN identifiers only.
     * @param correspondIdentifiers list of patient identifiers (PatientIdentifiers) from any of the supported assigning authorities
     * @return list of patient identifier strings for the provided patients where ICN assigning authority was found
     * @return an empty list if no identifiers are found from the ICN assigning authority
     */
    public List<String> getIcnIdentifiers(PatientIdentifiers correspondIdentifiers) {
        List<String> foundIcnIdentifiers = new ArrayList<>();
        for (PatientIdentifier correspondIdentifier : correspondIdentifiers) {
            if ("ICN".equalsIgnoreCase(correspondIdentifier.getAssigningAuthority())) {
                foundIcnIdentifiers.add(correspondIdentifier.getUniqueId());
            }
        }
        return foundIcnIdentifiers;
    }

    /**
     * Retrieves PatientIdentifier from the EDIPI assigning authority for a patient.
     * Uses the provided PatientIdentifier to lookup and return the EDIPI identifier.
     * @param patientIdentifier PatientIdentifier from any of the supported assigning authorities
     * @return a PatientIdentifier for the provided patient from the EDIPI assigning authority
     * @return null if no identifier is found for the EDIPI assigning authority
     */
    protected PatientIdentifier getEdipiIdentifier(final PatientIdentifier patientIdentifier) {
        final PatientIdentifiers corresponingIdentifiers = patientCorrelationGateway.getCorrespondIdentifiers(patientIdentifier);

        for (PatientIdentifier identifier : corresponingIdentifiers) {
            if ("EDIPI".equalsIgnoreCase(identifier.getAssigningAuthority())) {
                return identifier;
            }
        }
        return null;
    }

    /**
     * Attempts to retrieve PatientIdentifier from the EDIPI assigning authority for a patient.
     * Scheduling Manager currently only supports EDIPI identifiers. Regardless of assigning authority for the
     * current user session, a lookup is performed for the EDIPI identifier and if one is not found, it falls
     * back to the identifier for the current user session.
     * @param assigningAuthority String of the assigning authority for the provided patient id
     * @param patientId String of the unique id for the patient
     * @return a PatientIdentifier for the provided patient from the EDIPI assigning authority
     * @return a PatientIdentifier for the current session when no EDIPI identifier is found
     */
    public PatientIdentifier getTargetPatientIdentifier(String assigningAuthority, String patientId) {
        PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
        PatientIdentifier targetIdentifier = getEdipiIdentifier(patientIdentifier);
        if (targetIdentifier == null) {
            targetIdentifier = patientIdentifier;
        }
        return targetIdentifier;
    }

    /**
     * Attempts to retrieve PatientIdentifier for the provided assigning authority for a patient.
     * Uses the provided PatientIdentifier to lookup all patient identifiers for a patient and returns the identifier
     * for the desired assigning authority.
     * @param patientIdentifier PatientIdentifier from any of the supported assigning authorities
     * @param assigningAuthority String of the desired assigning authority
     * @return a PatientIdentifier for the provided patient from the desired assigning authority
     * @return null if no identifier is found from the desired assigning authority
     */
    public PatientIdentifier getCorrelatingPatientIdentifier(PatientIdentifier patientIdentifier, String assigningAuthority) {
        PatientIdentifiers correlatingIdentifiers = patientCorrelationGateway.getCorrespondIdentifiers(patientIdentifier);

        for(PatientIdentifier pid : correlatingIdentifiers) {
            if(pid.getAssigningAuthority().equals(assigningAuthority)) {
                return new PatientIdentifier(assigningAuthority, pid.getUniqueId());
            }
        }

        return null;
    }

    /**
     * Retrieves all identifiers for a patient.
     * Uses the provided patient identifier to look up patient identifier from assigning authorities for the
     * given patient.
     * @param patientIdentifier PatientIdentifier from any of the supported assigning authorities
     * @return a list of patient identifiers (PatientIdentifiers) for the provided patient
     */
    public PatientIdentifiers getCorrespondIdentifiers(PatientIdentifier patientIdentifier) {
        return patientCorrelationGateway.getCorrespondIdentifiers(patientIdentifier);
    }

    public String stripHash(String originalIcn) {
        int index = originalIcn.toLowerCase().indexOf("v");
        if (index <= 0)
            return originalIcn;

        return originalIcn.substring(0, index);
    }

	public PatientIdentifiers filterForEdipiAndIcn(PatientIdentifiers identifiers){
		PatientIdentifiers identifiersFiltered = new PatientIdentifiers();

		for(PatientIdentifier identifier : identifiers){
			boolean isIcn = identifier.getAssigningAuthority().equalsIgnoreCase("ICN");
			boolean isEdipi = identifier.getAssigningAuthority().equalsIgnoreCase("EDIPI");
			if( isIcn || isEdipi ){
				identifiersFiltered.add(identifier);
			}
		}

		return identifiersFiltered;
	}

}
